<?php
/*
 * Erstellt von Daniel Messer (2005)
 */

set_include_path(realpath(dirname(__FILE__)."/../../").":".get_include_path());

require_once "lib/configuration/Configuration.php";

/**
 * Simple heir of class 'Exception' to catch specific exceptions thrown by the XSLTemplate class or
 * an inherited class.
 * @package    stdlib
 * @subpackage template
 */
class XSLTemplate_Exception extends Exception {}

/**
 * This is class is a template engine to serve the ability of seperating html code and the display
 * logic from the php code and its application logic. It is based on the XML-dialect XSL/XSLT
 * provided as a standard by the W3 consortium. The output is mainly controlled by the integrated
 * XSLT-processor, so XSL-support is required for the PHP parser. This is provided by the libxslt-
 * project, see http://xmlsoft.org/XSLT/ for further details. The XSLTemplate class only cares about
 * the variables assigned to the template and generates an xml-file (datasheet) containing the
 * valaues for these variables. A template is an XSL-stylesheet. When parsing an instance of an
 * XSLT- processor is created and the stylesheet and the datasheet is attached to it. So all
 * constructs of formatting the output or implementing repetition is put into effect by the
 * eXtensible Stylesheet Language. This means you are using a standard and makes the generation of
 * templates faster than any other template engines without the loosing features.
 * 
 * @author Daniel Messer <daniel.messer@web.de>
 * @package stdlib
 * @subpackage template
 * @version 2.0
 */
class XSLTemplate
{
	/**
	 * @var Configuration Contains an instance of the Configuration class serving the acccess to the
	 * configuration file. This object is created when instantiating this class so the configuration
	 * is read only once. Changes to the configuration file during the usage of an XSLTemplate
	 * object will not take effect to it.
	 * @access protected
	 */
	protected $configurationCache;
	
	/**
	 * @var array Associative array of parameters which are processed to the XSL-stylesheet.
	 * @access protected
	 */
	protected $templateParameters = array();
	
	/**
	 * @var array Associative array containing the variables assigned to the template in arrays.
	 * Every variable is an array with variable name as the first element and the value of this
	 * variable as the second. The value can containg such pair-arrays again, unlimited nesting is
	 * possible.
	 * @access protected
	 */
	protected $templateVariables = array();
	
	/**
	 * @var DOMDocument Contains the XSL-stylesheet read into an DOMDocument.
	 * @access protected
	 */
	protected $xslStylesheet;
	
	/**
	 * @var DOMDocument The XML-sheet containing the variables as data resource for the XSL-
	 * stylesheet.
	 * @access protected
	 */
	protected $xmlDatasheet;
	
	/**
	 * The constructor expects the name of the template. The template name is the file name without
	 * file suffix. The template is loaded via the loading mechanism implemented in XSLTemplate-
	 * >loadTemplate().
	 * Furthermore the environment variables of GET and POST are added as homonymous nodes under the
	 * subnode 'input' of the root node 'master'. So if you want to adress the PHP variable $_POST
	 * ["userInput"] in the stylesheet you have to select it with XPath:
	 * /master/input/post/userInput.
	 * Cookie variables are not added for security reasons.
	 * 
	 * @param string $templateName The name of the template to load.
	 * @return void This method is a constructor and does not return anything
	 * @access public
	 */
	public function __construct($templateName)
	{	
		$this->configurationCache = Configuration::loadResource();	
		$this->xslStylesheet = $this->loadTemplate($templateName);
		
		require_once "lib/misc/InputFilter.php"; // do net set this in global context or further include statements with return value will fail!
		
		$inputFilter = new InputFilter();
		
		$this->__set("input", array("post" => self::transformNumericIndexes($inputFilter->post)));

		$this->__set("input", array("get" => self::transformNumericIndexes($inputFilter->get)));
	}
	
	/**
	 * This is the template loading mechanism. The path where templates are searched in, is
	 * constructed with the relative path from the 'template'-section in the configuration file.
	 * This path relates to the project root 2 directories higher than this script file.
	 * The following script loads the XSL-stylesheet from ../../tpl/test.xsl relative to the
	 * position of this script.
	 * 
	 * <code>
	 * <?php
	 * 
	 * $tpl = new XSLTemplate("test");
	 *  	
	 * ?>
	 * </code>
	 * 
	 * @param string $templateName The template to load.
	 * @return DOMDocument An instance of the DOMDocument class with the representing the loaded
	 * template.
	 */	
	protected function loadTemplate($templateName)
	{
		$templatePath = realpath(dirname(__FILE__)."/../../".$this->configurationCache->get("path", "template"))."/".$templateName.".xsl";
		
		if(!is_file($templatePath))
		{
			throw new XSLTemplate_Exception("Requested template '".$templateName."' not found in ".$templatePath);
		}
	
		$loadedDOM = DOMDocument::load($templatePath);
		
		if(!$loadedDOM)
		{
			throw new XSLTemplate_Exception("Error loading template '".$templateName."', maybe it is not valid");
		}
			
		return $loadedDOM;
	}
	
	/**
	 * This method allows you to import an external stylesheet into the one handled by the instance
	 * of this class. The template is loaded with the mechanism implemented in XSLTemplate-
	 * >loadTemplate() All nodes under the xsl:stylesheet-node (the root node of every XSL-
	 * stylesheet) except the xsl:output-element are imported attached to current XSL-stylesheet
	 * handled by an instance of this class. Note that other stylesheets can also be included with
	 * an appropriate xsl-statement in the master stylesheet.
	 * 
	 * @param string $templateName The name template which should be imported.
	 * @access public
	 * @return void This method does not return anything.
	 */
	public function importTemplate($templateName)
	{
		$importedTemplate = $this->loadTemplate($templateName);
		
		$sourceNode = $importedTemplate->getElementsByTagNameNS("http://www.w3.org/1999/XSL/Transform", "stylesheet")->item(0);
		
		if(is_null($sourceNode)) 
		{
			throw new XSLTemplate_Exception("Required xsl:stylesheet root node not found in imported template '".$templateName."'");
		}
		
		$destinationNode = $this->xslStylesheet->getElementsByTagNameNS("http://www.w3.org/1999/XSL/Transform", "stylesheet")->item(0);
		
		if(is_null($destinationNode)) 
		{
			throw new XSLTemplate_Exception("Required xsl:stylesheet root node not found in template '".$templateName."'");
		}
		
		foreach($sourceNode->childNodes as $childNode)
		{
			if(($childNode->prefix == "xsl") && $childNode->localName != "output")
			{
				$importedNode = $this->xslStylesheet->importNode($childNode, TRUE);
				$destinationNode->appendChild($importedNode);	
			}
		}
	}
	
	/**
	 * This interceptor method overloading the object is used to assign variables to the template.
	 * XSL supports loops by stating multiple nodes with the same name into the xml database. This
	 * is implemented in this method when you assign a variable twice. The following script...
	 * 
	 * <code>
	 * <?php
	 * 
	 * $tpl = new XSLTemplate("test");
	 * $tpl->testvar = value1;
	 * $tpl->testvar = value2;
	 * 
	 * echo $tpl->saveXMLData();
	 * 
	 * ?>
	 * </code>
	 * 
	 * ... produce the following xml sheet as a databse for the stylesheet:
	 * 
	 * <code>
	 * <?xml version="1.0" encoding="UTF-8" ?>
	 * <master>
	 * 		<testvar>value1</testvar>
	 * 		<testvar>value2</testvar>
	 * </master>
	 * </code>
	 * 
	 * To produce nested nodes for looping multiple datasets arrays can be used in 2 ways.
	 * 
	 * 1) Attaching assoicative arrays with keys as the node name and values as the node value.
	 * 2) Attaching arrays containing pair arrays. A pair array is a two field  array, with the name
	 * of the node as the first and the value of the node as the second element.
	 * 
	 * As node values can also be arrays of these two types unlimited nesting is possible. So the
	 * following script...
	 * 
	 * <code>
	 * <?php
	 * 
	 * $tpl = new XSLTemplate("test");
	 * 
	 * $tpl->testvar = array("var1" => "value1", "var2" => array("var2-1" => "value2-1", "var2-2"
	 * => "value2-2"), "var3" => "value3");
	 * 
	 * $tpl->testvar = array(array("var1", "value1"), array("var2", array("var2-1", "value2-1"),
	 * array("var2-2", "value2-2"), array("var3", "value3-3"))));
	 * 
	 * echo $tpl->saveXMLData();
	 * 
	 * ?>
	 * </code>
	 * 
	 * will produce the following output:
	 * 
	 * <code>
	 * <?xml version="1.0" encoding="UTF-8" ?>
	 * <master>
	 * 		<testvar>
	 * 			<var1>value1</var>
	 * 			<var2>
	 * 				<var2-1>value2-1</var2-1>
	 * 				<var2-2>value2-2</var2-2>
	 * 			</var2>
	 * 			<var3>value3</var>
	 * 		</testvar>
	 * 		<testvar>
	 * 			<var1>value1</var>
	 * 			<var2>
	 * 				<var2-1>value2-1</var2-1>
	 * 				<var2-2>value2-2</var2-2>
	 * 			</var2>
	 * 			<var3>value3</var>
	 * 		</testvar>
	 * </master>
	 * </code>
	 * 
	 * Both methods can be mixed.
	 * 
	 * @param string $varName The name of the variable to be set.
	 * @param string $varValue The value of the variable to be set.
	 * @access public
	 * @return void This method returns nothing.
	 */	
	public function __set($varName, $varValue)
	{
		if(is_array($varValue))
		{
			$this->templateVariables[] = array($varName, self::assoc2pair($varValue));
		}
		else
		{
			$this->templateVariables[] = array($varName, $varValue);
		}
	}
	
	/**
	 * This mthod allows to set/override an xsl:param-element. The mechanism to set a parameter is
	 * directly implemende in the XSLTProcessors class and executed when parsing the template.
	 * 
	 * @param string $parameterName The name of the parameter to be set.
	 * @param string $parameterValue The value of the parameter to be set.
	 * @access public
	 * @return void This method returns nothing.
	 */
	public function setParameter($parameterName, $parameterValue)
	{
		$this->templateParameters[$parameterName] = $parameterValue;
	}
	
	/**
	 * Saves the XSL-stylesheet in his current state.
	 * 
	 * @access public
	 * @return string The XSL-stylesheet.
	 */	
	public function saveXSL()
	{
		header("Content-type: text/html; charset=UTF-8");
		
		return $this->xslStylesheet->saveXML();
	}

	/**
	 * Parses and returns the current stylesheet attached with the datasheet and the parameters.
	 * 
	 * @access public
	 * @return string The generated output of the XSLT-processor.
	 */
	public function saveXML()
	{
		header("Content-type: text/html; charset=UTF-8");
		
		$this->saveXMLData();
		
		$proc = new XSLTProcessor();
		$proc->registerPhpFunctions();
		$proc->importStylesheet($this->xslStylesheet);
		
		foreach($this->templateParameters as $paramName => $paramValue)
		{
			$proc->setParameter("", $paramName, $paramValue);
		}

		return $proc->transformToXml($this->xmlDatasheet);
	}
	
	/**
	 * Generates and returns the xml datasheet.
	 * 
	 * @access public
	 * @return string The generated datasheet
	 */
	public function saveXMLData()
	{	
		header("Content-type: text/html; charset=UTF-8");
		
		$this->xmlDatasheet = new DOMDocument("1.0", "UTF-8");
		$this->xmlDatasheet->formatOutput = true;
				
		$rootNode = $this->xmlDatasheet->appendChild($this->xmlDatasheet->createElement("master"));
		
		$this->saveNode($rootNode, $this->templateVariables);
		
		return $this->xmlDatasheet->saveXML();
	}
	
	/**
	 * Recursivley convert array structures into xml structures. Used by XSLTemplate->saveXMLData().
	 * 
	 * @param DOMNode $destinationNode The node to which the generated xml structures should be
	 * appended.
	 * @param array $dataArray The source array.
	 * @access protected
	 * @return void The methods returns nothing. All necessaray changes are made directly to the
	 * destination node.
	 */
	protected function saveNode($destinationNode, $dataArray)
	{
		if(!($this->xmlDatasheet instanceof DOMDocument))
		{
			throw new XSLTemplate_Exception("There is no internal xml document");
		}
		
		if(!is_array($dataArray))
		{
			throw new XSLTemplate_Exception("Given data is not an array");
		}
		
		foreach($dataArray as $varArray)
		{
			$varName = $varArray[0];
			$varValue = $varArray[1];
			
			if(is_array($varValue))
			{
				if(!is_string($varName))
				{	
					throw new XSLTemplate_Exception("An associative array as template variable should contain string indices for converting into xml structure");
				}
				
				$dataNode = $this->xmlDatasheet->createElement($varName);
				
				$this->saveNode($dataNode, $varValue);
				
				$destinationNode->appendChild($dataNode);
			}
			else
			{
				$destinationNode->appendChild($this->xmlDatasheet->createElement($varName, utf8_encode($varValue)));
			}
		}
	}
	
	/**
	 * Recursivley converts an associative array into an pair array. Every key=>value relation is
	 * resolved to an array containing two elements: the key and the value. Nesting is possible.
	 * 
	 * @param array $assocArray The associative source array.
	 * @return array The generated pair array structure.
	 * @access protected
	 */	
	protected static function assoc2pair($assocArray)
	{
		if(!is_array($assocArray))
		{
			throw new XSLTemplate_Exception("Internal engine error, expected associative array for conversion to pair array");
		}
		
		$pairArray = array();
			
		foreach($assocArray as $key => $value)
		{	
			if(is_string($key))
			{
				if(is_array($value))
				{
					$pairArray[] = array($key, self::assoc2pair($value));
				}
				else
				{
					$pairArray[] = array($key, $value);
				}
			}
			elseif(is_array($value))
			{
				$pairArray[] = self::assoc2pair($value);
			}
			else
			{
				$pairArray[] = $value;
			}
		}
		
		return $pairArray;
	}
	
	/**
	 * This method is mainly used for transforming get/post values submitted via multiple-input
	 * elements (labeled with [] in name attribute). It solves the problem that such elements are
	 * transformed to an array with numeric indices which cannot be passed as variables to the
	 * template because xml nodes cannot be numbers only. The given array is recursivley transformed
	 * to an array containing arrays with the appriate values of the key and the element value.
	 * Passing this result array to a template will create nodes named 'element' under the array
	 * node containg one 'key' and one 'value' node. Unlimited nesting is possible.
	 * 
	 * TODO: Find a way to avoid the problem of having numeric indexes in get/post/cookie-data.
	 * 
	 * @param array $assocArray The associative source array.
	 * @return array The multidimensional array representing the new node-structures
	 * @access protected
	 */	
	public static function transformNumericIndexes($assocArray)
	{
		foreach($assocArray as $key => $value)
		{
			if(is_numeric($key))
			{
				if(is_array($value))
				{
					$value = self::transformNumericIndexes($value);
				}
				
				$assocArray[] = array("element", array("key" => $key, "value" => $value));
				unset($assocArray[$key]);
			}
			elseif(is_array($value))
			{
				$assocArray[$key] = self::transformNumericIndexes($value);
			}
		}
		
		return $assocArray;
	}
}
?>
