<?php


class XSLTemplate_Exception extends Exception {}

class XSLTemplate
{

	protected $templateParameters = array();

	protected $templateVariables = array();

	
	protected $xslStylesheet;

	
	protected $xmlDatasheet;


	public function __construct($templateName)
	{
		$this->xslStylesheet = $this->loadTemplate($templateName);
	}

	
	protected function loadTemplate($templateName)
	{
		$templatePath = $templateName.".xsl";

		if(!is_file($templatePath))
		{
			throw new XSLTemplate_Exception("Requested template '".$templateName."' not found in ".$templatePath);
		}

		$loadedDOM = DOMDocument::load($templatePath);

		if(!$loadedDOM)
		{
			throw new XSLTemplate_Exception("Error loading template '".$templateName."', maybe it is not valid");
		}

		return $loadedDOM;
	}


	public function importTemplate($templateName)
	{
		$importedTemplate = $this->loadTemplate($templateName);

		$sourceNode = $importedTemplate->getElementsByTagNameNS("http://www.w3.org/1999/XSL/Transform", "stylesheet")->item(0);

		if(is_null($sourceNode))
		{
			throw new XSLTemplate_Exception("Required xsl:stylesheet root node not found in imported template '".$templateName."'");
		}

		$destinationNode = $this->xslStylesheet->getElementsByTagNameNS("http://www.w3.org/1999/XSL/Transform", "stylesheet")->item(0);

		if(is_null($destinationNode))
		{
			throw new XSLTemplate_Exception("Required xsl:stylesheet root node not found in template '".$templateName."'");
		}

		foreach($sourceNode->childNodes as $childNode)
		{
			if(($childNode->prefix == "xsl") && $childNode->localName != "output")
			{
				$importedNode = $this->xslStylesheet->importNode($childNode, TRUE);
				$destinationNode->appendChild($importedNode);
			}
		}
	}

	
	public function setVariable($varName, $varValue, $attribute=false)
	{
		if(is_array($varValue))
		{
			$this->templateVariables[] = array($varName, self::assoc2pair($varValue), $attribute);
		}
		else
		{
			$this->templateVariables[] = array($varName, $varValue,  $attribute);
		}
	}

	
	public function setParameter($parameterName, $parameterValue)
	{
		$this->templateParameters[$parameterName] = $parameterValue;
	}

	
	public function getXSL()
	{
		
		return $this->xslStylesheet->saveXML();
	}

	

	public function fetch()
	{
		
		$this->getXMLData();

		$proc = new XSLTProcessor();
		$proc->registerPhpFunctions();
		$proc->importStylesheet($this->xslStylesheet);

		foreach($this->templateParameters as $paramName => $paramValue)
		{
			$proc->setParameter("", $paramName, $paramValue);
		}

		return $proc->transformToXml($this->xmlDatasheet);
	}

	


	public function getXMLData()
	{
		
		$this->xmlDatasheet = new DOMDocument("1.0", "UTF-8");
		$this->xmlDatasheet->formatOutput = true;

		$rootNode = $this->xmlDatasheet->appendChild($this->xmlDatasheet->createElement("master"));

		$this->saveNode($rootNode, $this->templateVariables);

		return $this->xmlDatasheet->saveXML();
	}

	


	protected function saveNode($destinationNode, $dataArray)
	{
		if(!($this->xmlDatasheet instanceof DOMDocument))
		{
			throw new XSLTemplate_Exception("There is no internal xml document");
		}

		if(!is_array($dataArray))
		{
			throw new XSLTemplate_Exception("Given data is not an array");
		}

		foreach($dataArray as $varArray)
		{
			$varName = $varArray[0];
			$varValue = $varArray[1];
			$varAttribute = $varArray[2];

			if(is_array($varValue))
			{
				if(!is_string($varName))
				{
					throw new XSLTemplate_Exception("An associative array as template variable should contain string indices for converting into xml structure");
				}

				$dataNode = $this->xmlDatasheet->createElement($varName);

				$this->saveNode($dataNode, $varValue);

				$destinationNode->appendChild($dataNode);
			}
			else
			{
				$destinationNode->appendChild($this->xmlDatasheet->createElement($varName, utf8_encode($varValue)));
				if($varAttribute != false)
				{
					foreach($varAttribute as $key => $val)
					{
						$destinationNode->setAttribute($key, $val);
					}
				}

			}
		}
	}



	protected static function assoc2pair($assocArray)
	{
		if(!is_array($assocArray))
		{
			throw new XSLTemplate_Exception("Internal engine error, expected associative array for conversion to pair array");
		}

		$pairArray = array();

		foreach($assocArray as $key => $value)
		{
			if(is_string($key))
			{
				if(is_array($value))
				{
					$pairArray[] = array($key, self::assoc2pair($value));
				}
				else
				{
					$pairArray[] = array($key, $value);
				}
			}
			elseif(is_array($value))
			{
				$pairArray[] = array('item' ,self::assoc2pair($value));
			}
			else
			{
				$pairArray[] = $value;
			}
		}

		return $pairArray;
	}

}
?>
